#include <msp430.h> // generic file for register name, I/O, ...

#include "RF1A.h"


// ##########################################################
// This definitions and delay_RF were added by Penko Bozhkov
// ##########################################################
#define TXtoIDLE_Time     10    // TX to IDLE, no calibration: ~1us   => 0.3us *10 = 3us
#define RXtoIDLE_Time     2     // RX to IDLE, no calibration: ~0.1us => 0.3*2 = 0.6us
#define IDLEtoRX_Time     300   // IDLE to RX, no calibration: 75.1us => 0.3*300 = 90us


/************************************************************************/
/*  Function name: delay_RF                                             */
/*  	Parameters                                                      */
/*          Input   :  p	                                            */
/*          Output  :  No	                                            */
/*	Action: Simple delay                                                */
/************************************************************************/
void delay_RF(volatile unsigned long p){
	while(p){p--;}    // delay_RF will take at least 6 MCU cycles(20Mhz/6=3.33MHz) 1/3.33Mhz = 0.3us
}

void owndelay(int del){
	int t;
	for (t=0 ; t<del; t++)
		__delay_cycles(100000);
}

unsigned char Strobe(unsigned char strobe)
{
  unsigned char statusByte;
  
  // Check for valid strobe command 
  if((strobe == 0xBD) || ((strobe >= 0x30) && (strobe <= 0x3D)))
  {
    RF1AIFCTL1 &= ~(RFSTATIFG);             // Clear the status read flag
    
    while( !(RF1AIFCTL1 & RFINSTRIFG)  );   // Wait for INSTRIFG
    RF1AINSTRB = strobe;                    // Write the strobe command    
    
    if(strobe != 0x30) while( !(RF1AIFCTL1 & RFSTATIFG) ); 
    statusByte = RF1ASTATB;         
  }  
  else
    return 0;                               // Invalid strobe was sent
  
  return statusByte;
}

unsigned char ReadSingleReg(unsigned char addr)
{
  unsigned char data_out;
  
  // Check for valid configuration register address, 0x3E refers to PATABLE
  if ((addr <= 0x2E) || (addr == 0x3E))
    // Send address + Instruction + 1 dummy byte (auto-read)
    RF1AINSTR1B = (addr | RF_SNGLREGRD);
  else
    // Send address + Instruction + 1 dummy byte (auto-read)
    RF1AINSTR1B = (addr | RF_STATREGRD);
  
  while (!(RF1AIFCTL1 & RFDOUTIFG) );
  data_out = RF1ADOUT1B;                    // Read data and clears the RFDOUTIFG

  return data_out;
}

void WriteSingleReg(unsigned char addr, unsigned char value)
{   
  while (!(RF1AIFCTL1 & RFINSTRIFG));       // Wait for the Radio to be ready for next instruction
  RF1AINSTRB = (addr | RF_REGWR);			// Send address + Instruction
  RF1ADINB = value; 						// Write data in

  __no_operation();
}


void ReadBurstReg(unsigned char addr, unsigned char *buffer, unsigned char count)
{
  unsigned int i;
  
  while (!(RF1AIFCTL1 & RFINSTRIFG));       // Wait for INSTRIFG
  RF1AINSTR1B = (addr | RF_REGRD);          // Send addr of first conf. reg. to be read 
                                            // ... and the burst-register read instruction
  for (i = 0; i < (count-1); i++)
  {
    while (!(RFDOUTIFG&RF1AIFCTL1));        // Wait for the Radio Core to update the RF1ADOUTB reg
    buffer[i] = RF1ADOUT1B;                 // Read DOUT from Radio Core + clears RFDOUTIFG
                                            // Also initiates auo-read for next DOUT byte
  }
  buffer[count-1] = RF1ADOUT0B;             // Store the last DOUT from Radio Core
}

void WriteBurstReg(unsigned char addr, unsigned char *buffer, unsigned char count)
{  
  // Write Burst works wordwise not bytewise - known errata
  unsigned char i;

  while (!(RF1AIFCTL1 & RFINSTRIFG));       // Wait for the Radio to be ready for next instruction
  RF1AINSTRW = ((addr | RF_REGWR)<<8 ) + buffer[0]; // Send address + Instruction

  for (i = 1; i < count; i++)
  {
    RF1ADINB = buffer[i];                   // Send data
    while (!(RFDINIFG & RF1AIFCTL1));       // Wait for TX to finish
  } 
  i = RF1ADOUTB;                            // Reset RFDOUTIFG flag which contains status byte
}

void WriteSmartRFReg(const unsigned char SmartRFSetting[][2], unsigned char size)
{
  unsigned char i;
  for (i=0; i < (size); i++)
    WriteSingleReg(SmartRFSetting[i][0], SmartRFSetting[i][1]);
}

void ResetRadioCore (void)
{
  Strobe(RF_SRES);                          // Reset the Radio Core
  Strobe(RF_SNOP);                          // Reset Radio Pointer
}

// With aim to change RF output power, make your choice here!!!
void WritePATable(void)
{
  unsigned char valueRead = 0;
  
  //while(valueRead != 0x03)    // Output Power: -30  [dBm]
  //while(valueRead != 0x25)    // Output Power: -12  [dBm]
  //while(valueRead != 0x2D)    // Output Power: -6   [dBm]
  //while(valueRead != 0x8D)    // Output Power:  0   [dBm]
  while(valueRead != 0xC3)    // Output Power:  10  [dBm]
  //while(valueRead != 0xC0)    // Output Power:  Maximum  [dBm]
  //while(valueRead != 0xC6)    // Output Power(default):  8.8  [dBm]
  {
    /* Write the power output to the PA_TABLE and verify the write operation.  */
    unsigned char i = 0; 

    /* wait for radio to be ready for next instruction */
    while( !(RF1AIFCTL1 & RFINSTRIFG));
    //RF1AINSTRW = 0x7E03; // PA Table write (burst), Output Power: -30  [dBm]
    //RF1AINSTRW = 0x7E25; // PA Table write (burst), Output Power: -12  [dBm]
    //RF1AINSTRW = 0x7E2D; // PA Table write (burst), Output Power: -6   [dBm]
    //RF1AINSTRW = 0x7E8D; // PA Table write (burst), Output Power: 0    [dBm]
    RF1AINSTRW = 0x7EC3; // PA Table write (burst), Output Power: 10   [dBm]
    //RF1AINSTRW = 0x7EC0; // PA Table write (burst), Output Power: Maximum  [dBm]
    //RF1AINSTRW = 0x7EC6; // PA Table write (burst), Output Power(default):  8.8  [dBm]
    
    /* wait for radio to be ready for next instruction */
    while( !(RF1AIFCTL1 & RFINSTRIFG));
      RF1AINSTR1B = RF_PATABRD;                 // -miguel read & write
    
    // Traverse PATABLE pointers to read 
    for (i = 0; i < 7; i++)
    {
      while( !(RF1AIFCTL1 & RFDOUTIFG));
      valueRead  = RF1ADOUT1B;     
    }
    while( !(RF1AIFCTL1 & RFDOUTIFG));
    valueRead  = RF1ADOUTB;
  }
}

void rf_transmit(unsigned char *buffer, unsigned char length)
{
  RF1AIES |= BIT9;                          
  RF1AIFG &= ~BIT9;                         // Clear pending interrupts
  RF1AIE |= BIT9;                           // Enable RFIFG9 TX end-of-packet interrupts
  
  /* RF1AIN_9 => Rising edge indicates SYNC sent/received and
   *             Falling edge indicates end of packet.
   *             Configure it to interrupt on falling edge.
   */   
  WriteBurstReg(RF_TXFIFOWR, buffer, length);
  
  Strobe( RF_STX );                         // Strobe STX
}

void ReceiveOn(void)
{  
  RF1AIFG &= ~BIT4;                         // Clear a pending interrupt
  RF1AIE  |= BIT4;                          // Enable the interrupt 
  
  // Previous state has been Tx
  Strobe( RF_SIDLE );
  delay_RF(TXtoIDLE_Time);
  Strobe( RF_SRX );
  delay_RF(IDLEtoRX_Time);
}

void ReceiveOff(void)
{
  RF1AIE &= ~BIT4;                          // Disable RX interrupts
  RF1AIFG &= ~BIT4;                         // Clear pending IFG
  
  // Previous state has been Rx
  Strobe( RF_SIDLE );
  delay_RF(RXtoIDLE_Time);
  Strobe( RF_SFRX);      /* Flush the receive FIFO of any residual data */

}
